<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

try {
    // Validate required fields
    $required_fields = ['customer_id', 'delivery_phone', 'delivery_address', 'scheduled_date'];
    $missing_fields = [];
    
    foreach ($required_fields as $field) {
        if (!isset($_POST[$field]) || empty($_POST[$field])) {
            $missing_fields[] = $field;
        }
    }
    
    if (!empty($missing_fields)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Missing required fields: ' . implode(', ', $missing_fields)
        ]);
        exit();
    }
    
    // Extract form data
    $customer_id = $_POST['customer_id'] === '' ? null : (int)$_POST['customer_id'];
    $assigned_to = $_POST['assigned_to'] === '' ? null : (int)$_POST['assigned_to'];
    $delivery_phone = trim($_POST['delivery_phone']);
    $delivery_address = trim($_POST['delivery_address']);
    $scheduled_date = $_POST['scheduled_date'];
    $delivery_notes = trim($_POST['delivery_notes'] ?? '');
    $delivery_fee = (float)($_POST['delivery_fee'] ?? 0);
    $payment_method = $_POST['payment_method'] ?? 'cash';
    $quotation_id = $_POST['quotation_id'] === '' ? null : (int)$_POST['quotation_id'];
    $quotation_reference = trim($_POST['quotation_reference'] ?? '');
    
    // Validate scheduled date
    $scheduled_timestamp = strtotime($scheduled_date);
    if (!$scheduled_timestamp) {
        echo json_encode(['success' => false, 'message' => 'Invalid scheduled date format']);
        exit();
    }
    
    // Check if scheduled date is in the future
    if ($scheduled_timestamp <= time()) {
        echo json_encode(['success' => false, 'message' => 'Scheduled date must be in the future']);
        exit();
    }
    
    // Validate customer exists if provided
    if ($customer_id) {
        $stmt = $pdo->prepare("SELECT id FROM customers WHERE id = ?");
        $stmt->execute([$customer_id]);
        if (!$stmt->fetch()) {
            echo json_encode(['success' => false, 'message' => 'Invalid customer selected']);
            exit();
        }
    }
    
    // Validate assigned user exists if provided
    if ($assigned_to) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE id = ?");
        $stmt->execute([$assigned_to]);
        if (!$stmt->fetch()) {
            echo json_encode(['success' => false, 'message' => 'Invalid delivery person selected']);
            exit();
        }
    }
    
    // Validate quotation exists if provided
    if ($quotation_id) {
        $stmt = $pdo->prepare("SELECT id, status FROM quotations WHERE id = ?");
        $stmt->execute([$quotation_id]);
        $quotation = $stmt->fetch();
        if (!$quotation) {
            echo json_encode(['success' => false, 'message' => 'Invalid quotation selected']);
            exit();
        }
        if ($quotation['status'] !== 'active') {
            echo json_encode(['success' => false, 'message' => 'Selected quotation is not active']);
            exit();
        }
    }
    
    // Generate delivery number
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM deliveries WHERE DATE(created_at) = CURDATE()");
    $stmt->execute();
    $today_count = $stmt->fetchColumn();
    $delivery_number = 'DEL' . date('Ymd') . str_pad($today_count + 1, 3, '0', STR_PAD_LEFT);
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Insert delivery record (handling existing table structure)
        $stmt = $pdo->prepare("
            INSERT INTO deliveries (
                delivery_number, customer_id, assigned_to, delivery_phone, 
                delivery_address, scheduled_date, delivery_notes, delivery_fee, 
                payment_method, status, quotation_id, quotation_reference, 
                created_by, created_at, updated_at
            ) VALUES (
                ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', ?, ?, ?, NOW(), NOW()
            )
        ");
        
        $stmt->execute([
            $delivery_number,
            $customer_id,
            $assigned_to,
            $delivery_phone,
            $delivery_address,
            $scheduled_date,
            $delivery_notes,
            $delivery_fee,
            $payment_method,
            $quotation_id,
            $quotation_reference,
            $_SESSION['user_id']
        ]);
        
        $delivery_id = $pdo->lastInsertId();
        
        // If quotation is provided, update quotation status to 'converted' (since delivery_scheduled doesn't exist)
        if ($quotation_id) {
            $stmt = $pdo->prepare("UPDATE quotations SET status = 'converted', updated_at = NOW() WHERE id = ?");
            $stmt->execute([$quotation_id]);
        }
        
        // Log the action (with error handling)
        try {
            $action = $quotation_id ? 'Created delivery from quotation' : 'Created new delivery';
            $stmt = $pdo->prepare("
                INSERT INTO activity_logs (user_id, action, details, ip_address, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $_SESSION['user_id'],
                $action,
                "Delivery #$delivery_number created" . ($quotation_id ? " from quotation #$quotation_reference" : ''),
                $_SERVER['REMOTE_ADDR'] ?? 'unknown'
            ]);
        } catch (Exception $e) {
            // Log error but don't fail the delivery creation
            error_log("Failed to log activity: " . $e->getMessage());
        }
        
        // Commit transaction
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => "Delivery #$delivery_number scheduled successfully!",
            'delivery_number' => $delivery_number,
            'delivery_id' => $delivery_id
        ]);
        
    } catch (Exception $e) {
        // Rollback transaction on error
        $pdo->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    error_log("Database error in create_delivery.php: " . $e->getMessage());
    error_log("SQL State: " . $e->getCode());
    error_log("Error Info: " . print_r($e->errorInfo, true));
    echo json_encode([
        'success' => false, 
        'message' => 'Database error occurred. Please try again.',
        'debug' => 'Check server logs for details'
    ]);
} catch (Exception $e) {
    error_log("Error in create_delivery.php: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    echo json_encode([
        'success' => false, 
        'message' => 'An error occurred. Please try again.',
        'debug' => 'Check server logs for details'
    ]);
}
?> 